"use client"

import * as React from "react"
import { useTranslations } from "next-intl"
import { Plus, Send } from "lucide-react"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Textarea } from "@/components/ui/textarea"
import StockRequestTable from "@/components/stock/StockRequestTable"
import type { StockRequestLine } from "@/components/stock/stock-request-types"

export default function StockRequestCreatePage() {
  const t = useTranslations()

  const [fromLocation, setFromLocation] = React.useState("")
  const [toLocation, setToLocation] = React.useState("")
  const [requestedBy, setRequestedBy] = React.useState("")
  const [notes, setNotes] = React.useState("")

  const [lines, setLines] = React.useState<StockRequestLine[]>([
    { id: crypto.randomUUID(), item: "", unit: "", requestedQty: 1 },
  ])

  function addLine() {
    setLines((prev) => [
      ...prev,
      { id: crypto.randomUUID(), item: "", unit: "", requestedQty: 1 },
    ])
  }

  function updateLine(id: string, patch: Partial<StockRequestLine>) {
    setLines((prev) =>
      prev.map((l) => (l.id === id ? { ...l, ...patch } : l))
    )
  }

  function removeLine(id: string) {
    setLines((prev) => prev.filter((line) => line.id !== id))
  }

  function submit() {
    console.log({
      fromLocation,
      toLocation,
      requestedBy,
      notes,
      lines,
    })
    // Replace with API call when ready
  }

  return (
    <div className="w-full p-4 md:p-6 lg:p-8 space-y-6">
      <div className="flex items-center justify-between">
        <h1 className="text-2xl font-semibold">{t("stockRequests.create")}</h1>
        <div className="flex gap-2">
          <Button onClick={addLine} variant="outline" size="sm">
            <Plus className="mr-2 h-4 w-4" />
            {t("stockRequests.addItem")}
          </Button>
          <Button onClick={submit} size="sm">
            <Send className="mr-2 h-4 w-4" />
            {t("stockRequests.submit")}
          </Button>
        </div>
      </div>

      <div className="rounded-md border bg-card p-4">
        <div className="grid grid-cols-12 gap-3">
          <Field label={t("stockRequests.fromLocation")} className="col-span-12 md:col-span-4">
            <Input className="h-9" value={fromLocation} onChange={(e) => setFromLocation(e.target.value)} />
          </Field>
          <Field label={t("stockRequests.toLocation")} className="col-span-12 md:col-span-4">
            <Input className="h-9" value={toLocation} onChange={(e) => setToLocation(e.target.value)} />
          </Field>
          <Field label={t("stockRequests.requestedBy")} className="col-span-12 md:col-span-4">
            <Input className="h-9" value={requestedBy} onChange={(e) => setRequestedBy(e.target.value)} />
          </Field>
          <Field label={t("stockRequests.notes")} className="col-span-12">
            <Textarea value={notes} onChange={(e) => setNotes(e.target.value)} />
          </Field>
        </div>
      </div>

      <StockRequestTable
        lines={lines}
        updateLine={updateLine}
        editable={true}
        showAdjusted={false}
        onRemove={removeLine} // ✅ Pass remove function
      />
    </div>
  )
}

function Field({
  label,
  className,
  children,
}: React.PropsWithChildren<{ label: string; className?: string }>) {
  return (
    <div className={className}>
      <label className="mb-1 block text-xs font-medium text-muted-foreground">{label}</label>
      {children}
    </div>
  )
}
