"use client"

import * as React from "react"
import { useTranslations } from "next-intl"
import { Input } from "@/components/ui/input"
import { Table, TableHeader, TableRow, TableHead, TableBody, TableCell } from "@/components/ui/table"
import { Button } from "@/components/ui/button"
import { PrintButton } from "../_components/report-helpers"

type ItemRow = {
  id: string
  name: string
  category: string
  unit: string
  qty: number
  minStock: number
  lastPurchase?: string
  lastCost: number
}

const seed: ItemRow[] = [
  { id: "i1", name: "Gloves", category: "Consumable", unit: "BOX", qty: 120, minStock: 50, lastPurchase: "2025-07-20", lastCost: 12 },
  { id: "i2", name: "Syringes", category: "Consumable", unit: "PK", qty: 10, minStock: 30, lastPurchase: "2025-07-29", lastCost: 6 },
]

export default function StockSummaryReport() {
  const t = useTranslations("Reports")
  const [search, setSearch] = React.useState("")

  const filtered = React.useMemo(() => {
    if (!search.trim()) return seed
    const q = search.toLowerCase()
    return seed.filter(r => (
      r.name.toLowerCase().includes(q) ||
      r.category.toLowerCase().includes(q) ||
      r.unit.toLowerCase().includes(q) ||
      r.lastPurchase?.includes(q) ||
      (r.qty.toString().includes(q)) ||
      (r.minStock.toString().includes(q))
    ))
  }, [search])

  const downloadCSV = () => {
    const headers = [
      t("summary.columns.name"),
      t("summary.columns.category"),
      t("summary.columns.unit"),
      t("summary.columns.availableQty"),
      t("summary.columns.minStock"),
      t("summary.columns.lastPurchaseDate"),
      t("summary.columns.totalValue"),
    ]

    const rows = filtered.map(r => [
      r.name,
      r.category,
      r.unit,
      r.qty,
      r.minStock,
      r.lastPurchase || "-",
      (r.qty * r.lastCost).toFixed(2),
    ])

    const csv = [
      headers.join(","),
      ...rows.map(row => row.map(field => `"${field}"`).join(",")),
    ].join("\n")

    const blob = new Blob([csv], { type: "text/csv" })
    const url = URL.createObjectURL(blob)
    const link = document.createElement("a")
    link.href = url
    link.download = "stock_summary.csv"
    document.body.appendChild(link)
    link.click()
    document.body.removeChild(link)
    URL.revokeObjectURL(url)
  }

  return (
    <div className="p-6 space-y-4">
      {/* Header */}
      <div className="flex justify-between items-center flex-wrap gap-4">
        <h1 className="text-3xl font-bold">{t("summary.title")}</h1>
        <div className="flex items-center gap-2">
          <Button onClick={downloadCSV} variant="outline">
            {t("common.export", { default: "Export CSV" })}
          </Button>
          <PrintButton label={t("common.print")} />
        </div>
      </div>

      {/* Search aligned to right */}
      <div className="flex justify-end">
        <Input
          className="h-9 text-sm w-full max-w-xs"
          placeholder={t("searchPlaceholder") || "Search..."}
          value={search}
          onChange={e => setSearch(e.target.value)}
        />
      </div>

      {/* Table */}
      <div className="overflow-hidden rounded-md border">
        <Table>
          <TableHeader>
            <TableRow>
              <TableHead>{t("summary.columns.name")}</TableHead>
              <TableHead>{t("summary.columns.category")}</TableHead>
              <TableHead>{t("summary.columns.unit")}</TableHead>
              <TableHead>{t("summary.columns.availableQty")}</TableHead>
              <TableHead>{t("summary.columns.minStock")}</TableHead>
              <TableHead>{t("summary.columns.lastPurchaseDate")}</TableHead>
              <TableHead>{t("summary.columns.totalValue")}</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {filtered.length ? filtered.map(r => (
              <TableRow key={r.id}>
                <TableCell>{r.name}</TableCell>
                <TableCell>{r.category}</TableCell>
                <TableCell>{r.unit}</TableCell>
                <TableCell className={r.qty <= r.minStock ? "text-red-600 font-medium" : ""}>{r.qty}</TableCell>
                <TableCell>{r.minStock}</TableCell>
                <TableCell>{r.lastPurchase || "-"}</TableCell>
                <TableCell>{(r.qty * r.lastCost).toFixed(2)}</TableCell>
              </TableRow>
            )) : (
              <TableRow>
                <TableCell colSpan={7} className="text-center text-muted-foreground">
                  {t("common.noData")}
                </TableCell>
              </TableRow>
            )}
          </TableBody>
        </Table>
      </div>
    </div>
  )
}
