"use client"

import * as React from "react"
import { useTranslations } from "next-intl"
import { Input } from "@/components/ui/input"
import { Button } from "@/components/ui/button"
import {
  Select,
  SelectTrigger,
  SelectValue,
  SelectContent,
  SelectItem,
} from "@/components/ui/select"
import {
  Table,
  TableHeader,
  TableRow,
  TableHead,
  TableBody,
  TableCell,
} from "@/components/ui/table"
import { PrintButton } from "../_components/report-helpers"

type Row = {
  id: string
  name: string
  category: string
  unit: string
  current: number
  reorder: number
}

const seed: Row[] = [
  { id: "i1", name: "Gloves", category: "Consumable", unit: "BOX", current: 8, reorder: 20 },
  { id: "i2", name: "Syringes", category: "Consumable", unit: "PK", current: 40, reorder: 30 },
]

export default function LowStockReport() {
  const t = useTranslations("Reports")

  const categories = React.useMemo(() => Array.from(new Set(seed.map((s) => s.category))), [])
  const units = React.useMemo(() => Array.from(new Set(seed.map((s) => s.unit))), [])

  const [cat, setCat] = React.useState("all")
  const [unit, setUnit] = React.useState("all")
  const [onlyLow, setOnlyLow] = React.useState(true)
  const [buffer, setBuffer] = React.useState(0)

  const filtered = seed.filter((r) => {
    if (cat !== "all" && r.category !== cat) return false
    if (unit !== "all" && r.unit !== unit) return false
    const threshold = r.reorder + buffer
    if (onlyLow && r.current > threshold) return false
    return true
  })

  const reset = () => {
    setCat("all")
    setUnit("all")
    setOnlyLow(true)
    setBuffer(0)
  }

  const downloadCSV = () => {
    const headers = [
      t("stock.columns.name"),
      t("stock.columns.category"),
      t("stock.columns.unit"),
      t("stock.columns.current"),
      t("stock.columns.reorder"),
    ]
    const rowsCsv = filtered.map(r =>
      [r.name, r.category, r.unit, r.current, r.reorder].map(val => `"${val}"`).join(",")
    )
    const csv = [headers.join(","), ...rowsCsv].join("\n")
    const blob = new Blob([csv], { type: "text/csv" })
    const url = URL.createObjectURL(blob)
    const a = document.createElement("a")
    a.href = url
    a.download = "low_stock_report.csv"
    document.body.appendChild(a)
    a.click()
    document.body.removeChild(a)
    URL.revokeObjectURL(url)
  }

  return (
    <div className="p-6 space-y-4">
      {/* Header */}
      <div className="flex items-center justify-between">
        <h1 className="text-3xl font-bold tracking-tight">{t("stock.title")}</h1>
        <div className="flex gap-2">
          <PrintButton label={t("common.print")} />
          <Button onClick={downloadCSV} variant="outline">
            {t("common.export")}
          </Button>
        </div>
      </div>

      {/* Filters */}
      <div className="print:hidden flex flex-wrap items-end gap-4">
        <div className="flex flex-col gap-1">
          <label className="text-sm font-semibold">{t("stock.filters.category")}</label>
          <Select value={cat} onValueChange={setCat}>
            <SelectTrigger className="h-9 w-40">
              <SelectValue placeholder="All" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">{t("common.all")}</SelectItem>
              {categories.map(c => (
                <SelectItem key={c} value={c}>{c}</SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>

        <div className="flex flex-col gap-1">
          <label className="text-sm font-semibold">{t("stock.columns.unit")}</label>
          <Select value={unit} onValueChange={setUnit}>
            <SelectTrigger className="h-9 w-32">
              <SelectValue placeholder="All" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">{t("common.all")}</SelectItem>
              {units.map(u => (
                <SelectItem key={u} value={u}>{u}</SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>

        <div className="flex flex-col gap-1">
          <label className="text-sm font-semibold">{t("stock.filters.buffer")}</label>
          <Input
            className="h-9 w-24"
            type="number"
            min={0}
            value={buffer}
            onChange={e => setBuffer(Number(e.target.value))}
          />
        </div>

        <div className="flex items-center gap-2 mt-6">
          <input
            id="onlyLow"
            type="checkbox"
            className="h-4 w-4"
            checked={onlyLow}
            onChange={e => setOnlyLow(e.target.checked)}
          />
          <label htmlFor="onlyLow" className="text-sm font-medium">
            {t("stock.filters.onlyLow")}
          </label>
        </div>

        <div className="ml-auto">
          <Button variant="outline" size="sm" onClick={reset}>
            {t("common.reset")}
          </Button>
        </div>
      </div>

      {/* Print-friendly filter summary */}
      <div className="hidden print:block text-sm text-gray-600">
        {t("stock.filters.summary")}:&nbsp;
        {t("stock.filters.category")}: {cat === "all" ? t("common.all") : cat};&nbsp;
        {t("stock.columns.unit")}: {unit === "all" ? t("common.all") : unit};&nbsp;
        {t("stock.filters.buffer")}: {buffer};&nbsp;
        {t("stock.filters.onlyLow")}: {onlyLow ? t("common.yes") : t("common.no")}
      </div>

      {/* Table or No Data */}
      {filtered.length === 0 ? (
        <div className="rounded-md border bg-red-50 p-4 text-center text-red-600">
          {t("common.noData")}
        </div>
      ) : (
        <div className="overflow-hidden rounded-md border">
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead>#</TableHead>
                <TableHead>{t("stock.columns.name")}</TableHead>
                <TableHead>{t("stock.columns.category")}</TableHead>
                <TableHead>{t("stock.columns.unit")}</TableHead>
                <TableHead>{t("stock.columns.current")}</TableHead>
                <TableHead>{t("stock.columns.reorder")}</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {filtered.map((r, i) => (
                <TableRow key={r.id}>
                  <TableCell>{i + 1}</TableCell>
                  <TableCell>{r.name}</TableCell>
                  <TableCell>{r.category}</TableCell>
                  <TableCell>{r.unit}</TableCell>
                  <TableCell
                    className={
                      r.current <= r.reorder + buffer ? "font-medium text-red-600" : ""
                    }
                  >
                    {r.current}
                  </TableCell>
                  <TableCell>{r.reorder}</TableCell>
                </TableRow>
              ))}
            </TableBody>
          </Table>
        </div>
      )}
    </div>
  )
}
