"use client"

import * as React from "react"
import { useTranslations } from "next-intl"
import { Input } from "@/components/ui/input"
import {
  Select,
  SelectTrigger,
  SelectValue,
  SelectContent,
  SelectItem,
} from "@/components/ui/select"
import {
  Table,
  TableHeader,
  TableRow,
  TableHead,
  TableBody,
  TableCell,
} from "@/components/ui/table"
import { PrintButton } from "../_components/report-helpers"
import { Button } from "@/components/ui/button"

type Row = {
  id: string
  item: string
  itemId: string
  batch: string
  category: string
  unit: string
  expiry: string
}

const items = [
  { id: "i1", name: "Gloves" },
  { id: "i2", name: "Syringes" },
]

const seed: Row[] = [
  {
    id: "b1",
    itemId: "i2",
    item: "Syringes",
    batch: "SR-22",
    category: "Consumable",
    unit: "PK",
    expiry: "2025-08-05",
  },
  {
    id: "b2",
    itemId: "i1",
    item: "Gloves",
    batch: "GV-09",
    category: "Consumable",
    unit: "BOX",
    expiry: "2025-11-15",
  },
]

function isExpired(d: string) {
  return new Date(d) < new Date(new Date().toDateString())
}

function isSoon(d: string, days = 30) {
  return !isExpired(d) && (new Date(d).getTime() - Date.now()) / 86400000 <= days
}

export default function ExpiryReport() {
  const t = useTranslations("Reports")

  const [item, setItem] = React.useState<string>("all")
  const [status, setStatus] = React.useState<"expired" | "soon" | "all">("expired")
  const [q, setQ] = React.useState("")

  const filtered = seed
    .filter(r => (item === "all" || r.itemId === item))
    .filter(r =>
      status === "all"
        ? true
        : status === "expired"
          ? isExpired(r.expiry)
          : isSoon(r.expiry)
    )
    .filter(r =>
      q
        ? `${r.item} ${r.batch}`.toLowerCase().includes(q.toLowerCase())
        : true
    )

  const reset = () => {
    setItem("all")
    setStatus("expired")
    setQ("")
  }

  function downloadCSV() {
    const headers = [
      t("expiry.columns.index"),
      t("expiry.columns.name"),
      t("expiry.columns.batch"),
      t("expiry.columns.category"),
      t("expiry.columns.unit"),
      t("expiry.columns.expiry"),
    ]

    const rows = filtered.map((r, idx) => [
      idx + 1,
      r.item,
      r.batch,
      r.category,
      r.unit,
      r.expiry,
    ])

    const csvContent = [
      headers.join(","),
      ...rows.map(row => row.map(String).map(field => `"${field}"`).join(",")),
    ].join("\n")

    const blob = new Blob([csvContent], { type: "text/csv" })
    const url = URL.createObjectURL(blob)
    const anchor = document.createElement("a")
    anchor.href = url
    anchor.download = "expiry_report.csv"
    document.body.appendChild(anchor)
    anchor.click()
    document.body.removeChild(anchor)
    URL.revokeObjectURL(url)
  }

  return (
    <div className="p-6 space-y-4">
      {/* Header */}
      <div className="flex items-center justify-between">
        <h1 className="text-3xl font-bold tracking-tight">{t("expiry.title")}</h1>
        <div className="flex gap-2">
          <PrintButton label={t("common.print")} />
          <Button onClick={downloadCSV} variant="outline">
            {t("common.export")}
          </Button>
        </div>
      </div>

      {/* Filters */}
      <div className="print:hidden flex flex-wrap items-end gap-4">
        <div className="flex flex-col gap-1">
          <label className="text-sm font-medium">{t("expiry.allItems")}</label>
          <Select value={item} onValueChange={setItem}>
            <SelectTrigger className="h-9 w-[200px] text-sm">
              <SelectValue placeholder={t("common.selectItem")} />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">{t("common.all")}</SelectItem>
              {items.map(i => (
                <SelectItem key={i.id} value={i.id}>{i.name}</SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>

        <div className="flex flex-col gap-1">
          <label className="text-sm font-medium">{t("expiry.status")}</label>
          <Select value={status} onValueChange={v => setStatus(v as any)}>
            <SelectTrigger className="h-9 w-[200px] text-sm">
              <SelectValue />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="expired">{t("expiry.statusExpired")}</SelectItem>
              <SelectItem value="soon">{t("expiry.statusSoon")}</SelectItem>
              <SelectItem value="all">{t("expiry.statusAll")}</SelectItem>
            </SelectContent>
          </Select>
        </div>

        <div className="flex flex-col gap-1 ml-auto">
          <label className="text-sm font-medium">{t("common.search")}</label>
          <Input
            className="h-9 w-64 text-sm"
            value={q}
            onChange={e => setQ(e.target.value)}
            placeholder={t("common.search")}
          />
        </div>

        <Button variant="outline" onClick={reset} className="h-9 text-sm">
          {t("common.reset")}
        </Button>
      </div>

      {/* Print Filter Summary */}
      <div className="hidden print:block text-sm text-muted-foreground">
        {t("expiry.allItems")}: {item === "all" ? t("common.all") : items.find(i => i.id === item)?.name || "—"};&nbsp;
        {t("expiry.status")}: {t(`expiry.status${status.charAt(0).toUpperCase() + status.slice(1)}`)}
      </div>

      {/* Table */}
      <div className="overflow-hidden rounded-md border">
        <Table>
          <TableHeader>
            <TableRow>
              <TableHead>{t("expiry.columns.index")}</TableHead>
              <TableHead>{t("expiry.columns.name")}</TableHead>
              <TableHead>{t("expiry.columns.batch")}</TableHead>
              <TableHead>{t("expiry.columns.category")}</TableHead>
              <TableHead>{t("expiry.columns.unit")}</TableHead>
              <TableHead>{t("expiry.columns.expiry")}</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {filtered.length ? (
              filtered.map((r, idx) => (
                <TableRow key={r.id}>
                  <TableCell>{idx + 1}</TableCell>
                  <TableCell>{r.item}</TableCell>
                  <TableCell>{r.batch}</TableCell>
                  <TableCell>{r.category}</TableCell>
                  <TableCell>{r.unit}</TableCell>
                  <TableCell
                    className={
                      isExpired(r.expiry)
                        ? "text-red-600 font-medium"
                        : isSoon(r.expiry)
                        ? "text-yellow-600 font-medium"
                        : ""
                    }
                  >
                    {r.expiry}
                  </TableCell>
                </TableRow>
              ))
            ) : (
              <TableRow>
                <TableCell colSpan={6} className="text-center text-muted-foreground">
                  {t("common.noData")}
                </TableCell>
              </TableRow>
            )}
          </TableBody>
        </Table>
      </div>
    </div>
  )
}
