'use client';

import React, { useState } from 'react';
import Link from 'next/link';
import { usePathname } from 'next/navigation';
import { cn } from '@/lib/utils';
import {
  Package,
  Boxes,
  Truck,
  Store,
  Receipt,
  BarChart3,
  ClipboardList,
  ChevronRight,
  Gauge,
} from "lucide-react";
import { useTranslations } from 'next-intl';




export default function AccountingLayout({
  children,
}: {
  children: React.ReactNode;
}) {
  const pathname = usePathname();
  
  const t = useTranslations("InventoryNavbar")
  
  const inventoryNavItems = [
  {
    title: t("items"),
    icon: Package,
    children: [
      { title: t("allItems"), href: "/inventory/items" },
      { title: t("addNewItem"), href: "/inventory/items/create" },
      { title: t("units"), href: "/inventory/units" },
      { title: t("itemCategory"), href: "/inventory/item-categories" },
      { title: t("itemTypes"), href: "/inventory/item-types" },
    ],
  },
  {
    title: t("stock"),
    icon: Boxes,
    children: [
      { title: t("stockHistory"), href: "/inventory/stock/history" },
      { title: t("stockAdjustment"), href: "/inventory/stock/adjustment" },
      { title: t("stockTransfer"), href: "/inventory/stock/transfer" },
      { title: t("stockUsage"), href: "/inventory/stock/usage" },
      { title: t("stockRequest"), href: "/inventory/stock/request" }
    ],
  },
  {
    title: t("suppliers"),
    icon: Truck,
    children: [
      { title: t("allSuppliers"), href: "/inventory/suppliers" },
    ],
  },
  {
    title: t("stores"),
    icon: Store,
    children: [
      { title: t("allStores"), href: "/inventory/stores" }],
  },
  {
    title: t("bills"),
    icon: Receipt,
    children: [
      { title: t("allBills"), href: "/inventory/bills" },
      { title: t("createBill"), href: "/inventory/bills/new" },
      { title: t("payBill"), href: "/inventory/bills/pay" },
    ],
  },
  {
    title: t("reports"),
    icon: BarChart3,
    children: [
      { title: t("stockSummary"), href: "/inventory/reports/summary" },
      { title: t("stockLedger"), href: "/inventory/reports/ledger" },
      { title: t("usageReport"), href: "/inventory/reports/usage" },
      { title: t("lowStock"), href: "/inventory/reports/stock" },
      { title: t("stockAging"), href: "/inventory/reports/aging" },
      { title: t("adjustmentReport"), href: "/inventory/reports/adjustments" },
      { title: t("expiryReport"), href: "/inventory/reports/expiry" },
    ],
  },
  {
    title: t("purchaseOrders"),
    icon: ClipboardList,
    children: [
      { title: t("allPurchaseOrders"), href: "/inventory/purchase-orders" },
      { title: t("createPurchaseOrder"), href: "/inventory/purchase-orders/create" },
    ],
  },
];

  const [openSections, setOpenSections] = useState<string[]>([]);

  const toggleSection = (title: string) => {
    setOpenSections((prev: any) =>
      prev.includes(title)
        ? prev.filter((t: any) => t !== title)
        : [...prev, title]
    );
  };

  return (
    <div className="flex h-full">
      {/* Sidebar Navigation */}
      <div className="w-64 border-r bg-gray-50/50 p-4 space-y-2">
        <div className="mb-6">
          <h2 className="text-lg font-semibold px-3">{t("inventory")}</h2>
          <p className="text-sm text-muted-foreground px-3">{t("inventory_manage")}</p>
        </div>

        <nav className="space-y-1">
          <div key={t("dashboard")} className="space-y-1">
                <Link
                  href="/inventory" 
                  className={cn(
                    "flex w-full items-center gap-2 px-3 py-2 text-sm font-medium transition-all rounded-lg hover:bg-gray-100",
                  )}
                >
                  <Gauge className={cn("h-4 w-4")} />
                  <span className="flex-1 text-left">{t("dashboard")}</span>
                </Link>
                </div>
          {inventoryNavItems.map((section) => {
            const isOpen = openSections.includes(section.title);
            const Icon = section.icon;


            return (
              <div key={section.title} className="space-y-1">
                <button
                  onClick={() => toggleSection(section.title)}
                  className={cn(
                    "flex w-full items-center gap-2 px-3 py-2 text-sm font-medium transition-all rounded-lg hover:bg-gray-100",
                  )}
                >
                  <Icon className={cn("h-4 w-4")} />
                  <span className="flex-1 text-left">{section.title}</span>
                  <span
                    className={cn(
                      "transition-transform",
                      isOpen ? "rotate-90" : "rotate-0"
                    )}
                  >
                    <ChevronRight className="h-4 w-4" />
                  </span>
                </button>

              
                <div
                  className={cn(
                    "overflow-hidden transition-all duration-300",
                    isOpen ? "max-h-[500px]" : "max-h-0"
                  )}
                >
                  {section.children?.map((item) => {
                    const isActive =
                      pathname === item.href;

                    return (
                      <Link
                        key={item.href}
                        href={item.href}
                        className={cn(
                          "ml-6 flex items-center gap-2 rounded-lg px-3 py-2 text-sm transition-all hover:bg-gray-100",
                          isActive && "bg-gray-100 font-medium text-primary"
                        )}
                      >
                        <span>{item.title}</span>
                      </Link>
                    );
                  })}
                </div>
              </div>
            );
          })}
        </nav>

      </div>

      {/* Main Content */}
      <div className="flex-1 overflow-auto">
        <div className="container mx-auto p-6">
          {children}
        </div>
      </div>
    </div>
  );
}