"use client"

import * as React from "react"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Table, TableHeader, TableRow, TableHead, TableBody, TableCell } from "@/components/ui/table"
import { Select, SelectTrigger, SelectValue, SelectContent, SelectItem } from "@/components/ui/select"

type Bill = { id: string; no: string; supplierId: string; supplierName: string; date: string; total: number; paid: number; status: "OPEN"|"PAID" }
type Supplier = { id: string; name: string }

const suppliers: Supplier[] = [
  { id: "s1", name: "Tempcon Trading" },
  { id: "s2", name: "Nice Touch" },
]

const billsSeed: Bill[] = [
  { id: "b1", no: "B-0001", supplierId: "s1", supplierName: "Tempcon Trading", date: "2025-08-01", total: 500, paid: 200, status: "OPEN" },
  { id: "b2", no: "B-0002", supplierId: "s1", supplierName: "Tempcon Trading", date: "2025-08-03", total: 150, paid: 0, status: "OPEN" },
  { id: "b3", no: "B-0003", supplierId: "s2", supplierName: "Nice Touch", date: "2025-08-05", total: 300, paid: 300, status: "PAID" },
]

export default function PayBillsPage() {
  const [supplierId, setSupplierId] = React.useState<string>("")
  const [reference, setReference] = React.useState("")
  const [note, setNote] = React.useState("")
  const [rows, setRows] = React.useState(billsSeed)
  const [payMap, setPayMap] = React.useState<Record<string, number>>({})
  const [selectedIds, setSelectedIds] = React.useState<Set<string>>(new Set())

  const openBills = rows.filter(b => b.supplierId === supplierId && b.status === "OPEN")
  const totalToPay = openBills.reduce((sum, b) => sum + (payMap[b.id] || 0), 0)

  function setPay(id: string, val: number, max: number) {
    const v = Math.max(0, Math.min(val, max))
    setPayMap(m => ({ ...m, [id]: v }))
  }

  function toggleSelect(id: string, due: number) {
    setSelectedIds(prev => {
      const next = new Set(prev)
      if (next.has(id)) {
        next.delete(id)
        setPayMap(p => ({ ...p, [id]: 0 }))
      } else {
        next.add(id)
        setPayMap(p => ({ ...p, [id]: due }))
      }
      return next
    })
  }

  function toggleAll(selectAll: boolean) {
    if (selectAll) {
      const newIds = new Set(openBills.map(b => b.id))
      const newPayMap: Record<string, number> = {}
      for (const b of openBills) {
        newPayMap[b.id] = b.total - b.paid
      }
      setSelectedIds(newIds)
      setPayMap(newPayMap)
    } else {
      setSelectedIds(new Set())
      setPayMap({})
    }
  }

  function savePayment() {
    if (!supplierId || totalToPay <= 0) return
    setRows(prev => prev.map(b => {
      if (!payMap[b.id]) return b
      const newPaid = Math.min(b.total, b.paid + payMap[b.id])
      return { ...b, paid: newPaid, status: newPaid >= b.total ? "PAID" : "OPEN" }
    }))
    setPayMap({})
    setReference("")
    setNote("")
    setSelectedIds(new Set())
    alert("Payment saved (demo).")
  }

  return (
    <div className="p-2 space-y-6">
      <h1 className="text-3xl font-bold tracking-tight">Pay Bills</h1>

      <div className="grid grid-cols-12 gap-x-6 gap-y-4">
        <Field label="Select Supplier" className="col-span-12 md:col-span-4">
          <Select value={supplierId} onValueChange={setSupplierId}>
            <SelectTrigger className="h-10 w-full">
              <SelectValue placeholder="Select Supplier" />
            </SelectTrigger>
            <SelectContent>
              {suppliers.map(s => <SelectItem key={s.id} value={s.id}>{s.name}</SelectItem>)}
            </SelectContent>
          </Select>
        </Field>

        <Field label="Reference" className="col-span-12 md:col-span-4">
          <Input
            className="h-10 w-full"
            placeholder="Enter Reference"
            value={reference}
            onChange={e => setReference(e.target.value)}
          />
        </Field>

        <Field label="Note" className="col-span-12 md:col-span-4">
          <textarea
            className="h-10 w-full resize-none rounded-md border bg-background px-3 py-2 text-sm shadow-sm"
            placeholder="Note if any"
            value={note}
            onChange={e => setNote(e.target.value)}
          />
        </Field>
      </div>

      <div className="rounded-md border bg-muted/20 p-6 text-center text-muted-foreground" hidden={!!supplierId}>
        Please Select Supplier
      </div>

      {supplierId && (
        <div className="space-y-3">
          <div className="overflow-hidden rounded-md border">
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>
                    <input
                      type="checkbox"
                      checked={selectedIds.size === openBills.length}
                      onChange={e => toggleAll(e.target.checked)}
                    />
                  </TableHead>
                  <TableHead>Bill No</TableHead>
                  <TableHead>Date</TableHead>
                  <TableHead>Total</TableHead>
                  <TableHead>Paid</TableHead>
                  <TableHead>Amount Due</TableHead>
                  <TableHead>Pay Now</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {openBills.length ? openBills.map(b => {
                  const due = Math.max(0, b.total - b.paid)
                  return (
                    <TableRow key={b.id}>
                      <TableCell>
                        <input
                          type="checkbox"
                          checked={selectedIds.has(b.id)}
                          onChange={() => toggleSelect(b.id, due)}
                        />
                      </TableCell>
                      <TableCell>{b.no}</TableCell>
                      <TableCell>{b.date}</TableCell>
                      <TableCell>{b.total.toFixed(2)}</TableCell>
                      <TableCell>{b.paid.toFixed(2)}</TableCell>
                      <TableCell className="font-medium">{due.toFixed(2)}</TableCell>
                      <TableCell className="w-48">
                        <Input
                          className="h-10"
                          type="number"
                          inputMode="decimal"
                          min={0}
                          max={due}
                          step="0.01"
                          value={payMap[b.id] ?? 0}
                          onChange={e => setPay(b.id, Number(e.target.value), due)}
                        />
                      </TableCell>
                    </TableRow>
                  )
                }) : (
                  <TableRow>
                    <TableCell colSpan={7} className="text-center text-muted-foreground">
                      No open bills
                    </TableCell>
                  </TableRow>
                )}
              </TableBody>
            </Table>
          </div>

          <div className="flex items-center justify-between">
            <div className="text-sm text-muted-foreground">Total to pay:</div>
            <div className="text-xl font-semibold tabular-nums">{totalToPay.toFixed(2)}</div>
          </div>

          <div className="flex justify-end gap-2">
            <Button variant="outline" onClick={() => { setPayMap({}); setSelectedIds(new Set()); setSupplierId("") }}>
              Cancel
            </Button>
            <Button onClick={savePayment} disabled={totalToPay <= 0}>
              Save Payment
            </Button>
          </div>
        </div>
      )}
    </div>
  )
}

function Field({ label, className, children }: React.PropsWithChildren<{ label: string; className?: string }>) {
  return (
    <div className={["min-w-0", className].filter(Boolean).join(" ")}>
      <label className="mb-1 block text-sm font-semibold">{label}</label>
      {children}
    </div>
  )
}
