"use client"

import * as React from "react"
import { useRouter } from "next/navigation"
import SubmitButton from "@/components/submit-button"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import {
  Table, TableHeader, TableRow, TableHead, TableBody, TableCell
} from "@/components/ui/table"
import {
  Select, SelectTrigger, SelectValue, SelectContent, SelectItem
} from "@/components/ui/select"
import {
  Dialog, DialogContent, DialogHeader, DialogTitle, DialogFooter, DialogTrigger
} from "@/components/ui/dialog"
import { Plus } from "lucide-react"

type BillLine = {
  id: string
  item: string
  batch?: string
  qty: number
  unit: string
  cost: number
  expiry?: string
  discount: number
  availableQty?: number
}

const suppliers = [
  { id: "s1", name: "Tempcon Trading" },
  { id: "s2", name: "Nice Touch" },
]
const items = ["Gloves", "Syringes", "Mask", "Sanitizer"]
const units = ["PC", "BOX", "PK"]

export default function AddBillPage() {
  const router = useRouter()
  const [paymentTerms, setPaymentTerms] = React.useState([
    { id: "due", label: "Due on Receipt", days: 0 },
    { id: "net30", label: "Net 30", days: 30 },
    { id: "net60", label: "Net 60", days: 60 },
  ])
  const [termDialogOpen, setTermDialogOpen] = React.useState(false)
  const [newTerm, setNewTerm] = React.useState({ label: "", days: 0 })
  const [form, setForm] = React.useState({
    vendorId: "",
    billDate: new Date().toISOString().slice(0, 10),
    dueDate: new Date().toISOString().slice(0, 10),
    refNo: "",
    paymentTermId: "",
    status: "Approved",
    notes: "",
    tax: 0,
    discount: 0,
    attach: null as File | null,
    lines: [] as BillLine[],
  })

  React.useEffect(() => {
    if (form.lines.length === 0) addLine()
  }, [])

  const subtotal = form.lines.reduce((s, l) => s + l.qty * l.cost - l.discount, 0)
  const total = Math.max(0, subtotal + form.tax - form.discount)
  const canSave = form.vendorId && form.lines.every(l => l.item && l.unit && l.qty > 0)

  function addLine() {
    setForm(f => ({
      ...f,
      lines: [...f.lines, {
        id: crypto.randomUUID(),
        item: "",
        batch: "",
        qty: 1,
        unit: "",
        cost: 0,
        expiry: "",
        discount: 0,
        availableQty: 0,
      }],
    }))
  }

  function setLine(id: string, patch: Partial<BillLine>) {
    setForm(f => ({
      ...f,
      lines: f.lines.map(l => (l.id === id ? { ...l, ...patch } : l)),
    }))
  }

  function removeLine(id: string) {
    setForm(f => ({
      ...f,
      lines: f.lines.filter(l => l.id !== id),
    }))
  }

  function save() {
    if (!canSave) return
    router.push("/bills")
  }

  function addPaymentTerm() {
    if (!newTerm.label.trim()) return
    const id = `t_${Date.now()}`
    const t = { id, label: newTerm.label.trim(), days: newTerm.days }
    setPaymentTerms(prev => [...prev, t])
    setForm(f => ({ ...f, paymentTermId: id }))
    setNewTerm({ label: "", days: 0 })
    setTermDialogOpen(false)
  }

  return (
    <div className="w-full p-2 space-y-6">
      <h1 className="text-3xl font-semibold">Create New Bill</h1>

      <div className="rounded-md border bg-card p-4 space-y-6">
        {/* Top Form Fields */}
        <div className="flex flex-wrap items-end gap-3">
          <Field label="Vendor" className="min-w-[110px]">
            <Select
              value={form.vendorId}
              onValueChange={v => setForm(f => ({ ...f, vendorId: v }))}
            >
              <SelectTrigger className="h-9 text-sm">
                <SelectValue placeholder="Select Vendor" />
              </SelectTrigger>
              <SelectContent>
                {suppliers.map(s => (
                  <SelectItem key={s.id} value={s.id}>{s.name}</SelectItem>
                ))}
              </SelectContent>
            </Select>
          </Field>

          <Field label="Bill Date" className="w-[110px]">
            <Input
              type="date"
              className="h-9 text-sm"
              value={form.billDate}
              onChange={e => setForm(f => ({ ...f, billDate: e.target.value }))}
            />
          </Field>

          <Field label="Due Date" className="w-[110px]">
            <Input
              type="date"
              className="h-9 text-sm"
              value={form.dueDate}
              onChange={e => setForm(f => ({ ...f, dueDate: e.target.value }))}
            />
          </Field>

          <Field label="Ref (Vendor)" className="w-[130px]">
            <Input
              className="h-9 text-sm"
              value={form.refNo}
              onChange={e => setForm(f => ({ ...f, refNo: e.target.value }))}
            />
          </Field>

          <Field label="Payment Term" className="w-[160px]">
            <div className="flex items-center gap-1">
              <Select
                value={form.paymentTermId}
                onValueChange={v => setForm(f => ({ ...f, paymentTermId: v }))}
              >
                <SelectTrigger className="h-9 text-sm">
                  <SelectValue placeholder="Select Term" />
                </SelectTrigger>
                <SelectContent>
                  {paymentTerms.map(pt => (
                    <SelectItem key={pt.id} value={pt.id}>
                      {pt.label}{pt.days ? ` (${pt.days}d)` : ""}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
              <Dialog open={termDialogOpen} onOpenChange={setTermDialogOpen}>
                <DialogTrigger asChild>
                  <Button variant="outline" size="icon" className="h-9 w-9"><Plus /></Button>
                </DialogTrigger>
                <DialogContent className="sm:max-w-md">
                  <DialogHeader><DialogTitle>Add Payment Term</DialogTitle></DialogHeader>
                  <div className="grid grid-cols-12 gap-3">
                    <div className="col-span-12">
                      <label className="text-sm font-semibold mb-1 block">Label</label>
                      <Input
                        className="h-9 text-sm"
                        placeholder="e.g., Net 30"
                        value={newTerm.label}
                        onChange={e => setNewTerm(t => ({ ...t, label: e.target.value }))}
                      />
                    </div>
                    <div className="col-span-12">
                      <label className="text-sm font-semibold mb-1 block">Days</label>
                      <Input
                        className="h-9 text-sm"
                        type="number"
                        placeholder="e.g., 30"
                        value={newTerm.days}
                        onChange={e => setNewTerm(t => ({ ...t, days: Number(e.target.value) }))}
                      />
                    </div>
                  </div>
                  <DialogFooter className="justify-end">
                    <Button variant="outline" onClick={() => setTermDialogOpen(false)}>Cancel</Button>
                    <Button onClick={addPaymentTerm} disabled={!newTerm.label.trim()}>Add</Button>
                  </DialogFooter>
                </DialogContent>
              </Dialog>
            </div>
          </Field>

          <Field label="Status" className="w-[120px]">
            <Select
              value={form.status}
              onValueChange={v => setForm(f => ({ ...f, status: v }))}
            >
              <SelectTrigger className="h-9 text-sm">
                <SelectValue placeholder="Approved" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="Approved">Approved</SelectItem>
                <SelectItem value="Pending">Pending</SelectItem>
                <SelectItem value="Paid">Paid</SelectItem>
              </SelectContent>
            </Select>
          </Field>
        </div>

        {/* Items Table */}
        <div className="mt-6">
          <div className="mb-2 text-xl font-semibold">Add Items</div>
          <div className="overflow-x-auto rounded-md border">
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="min-w-[120px]">Item</TableHead>
                  <TableHead className="min-w-[80px]">Batch</TableHead>
                  <TableHead className="min-w-[100px]">Available Qty</TableHead>
                  <TableHead className="min-w-[60px]">Qty</TableHead>
                  <TableHead className="min-w-[80px]">Unit</TableHead>
                  <TableHead className="min-w-[80px]">Cost</TableHead>
                  <TableHead className="min-w-[100px]">Expiry</TableHead>
                  <TableHead className="min-w-[80px]">Discount</TableHead>
                  <TableHead className="min-w-[80px]">Total</TableHead>
                  <TableHead className="w-20"></TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {form.lines.map(l => {
                  const lineTotal = l.qty * l.cost - l.discount
                  return (
                    <TableRow key={l.id}>
                      <TableCell>
                        <Select
                          value={l.item}
                          onValueChange={v => setLine(l.id, { item: v })}
                        >
                          <SelectTrigger className="h-9 text-sm">
                            <SelectValue placeholder="Item" />
                          </SelectTrigger>
                          <SelectContent>
                            {items.map(it => <SelectItem key={it} value={it}>{it}</SelectItem>)}
                          </SelectContent>
                        </Select>
                      </TableCell>
                      <TableCell>
                        <Input
                          className="h-9 text-sm"
                          value={l.batch || ""}
                          onChange={e => setLine(l.id, { batch: e.target.value })}
                        />
                      </TableCell>
                      <TableCell>
                        <Input
                          className="h-9 text-sm bg-gray-100 disabled:bg-gray-200 disabled:text-gray-500"
                          type="number"
                          value={l.availableQty || 0}
                          readOnly
                          disabled
                        />
                      </TableCell>
                      <TableCell>
                        <Input
                          className="h-9 text-sm"
                          type="number"
                          value={l.qty}
                          onChange={e => setLine(l.id, { qty: Number(e.target.value) })}
                        />
                      </TableCell>
                      <TableCell>
                        <Select
                          value={l.unit}
                          onValueChange={v => setLine(l.id, { unit: v })}
                        >
                          <SelectTrigger className="h-9 text-sm">
                            <SelectValue placeholder="Unit" />
                          </SelectTrigger>
                          <SelectContent>
                            {units.map(u => <SelectItem key={u} value={u}>{u}</SelectItem>)}
                          </SelectContent>
                        </Select>
                      </TableCell>
                      <TableCell>
                        <Input
                          className="h-9 text-sm"
                          type="number"
                          value={l.cost}
                          onChange={e => setLine(l.id, { cost: Number(e.target.value) })}
                        />
                      </TableCell>
                      <TableCell>
                        <Input
                          className="h-9 text-sm"
                          type="date"
                          value={l.expiry || ""}
                          onChange={e => setLine(l.id, { expiry: e.target.value })}
                        />
                      </TableCell>
                      <TableCell>
                        <Input
                          className="h-9 text-sm"
                          type="number"
                          value={l.discount}
                          onChange={e => setLine(l.id, { discount: Number(e.target.value) })}
                        />
                      </TableCell>
                      <TableCell className="font-medium">{lineTotal.toFixed(2)}</TableCell>
                      <TableCell className="text-right">
                        <Button variant="outline" size="sm" onClick={() => removeLine(l.id)}>✕</Button>
                      </TableCell>
                    </TableRow>
                  )
                })}
              </TableBody>
            </Table>
          </div>
          <Button variant="secondary" className="mt-3" onClick={addLine}>Add Line</Button>
        </div>

        {/* Notes & Totals */}
        <div className="grid grid-cols-12 gap-6 p-4">
          <Field label="Notes" className="col-span-12 md:col-span-6">
            <textarea
              rows={4}
              className="w-full rounded-md border bg-background px-3 py-2 text-sm shadow-sm"
              value={form.notes}
              onChange={e => setForm(f => ({ ...f, notes: e.target.value }))}
            />
          </Field>
          <div className="col-span-12 md:col-span-6 space-y-3">
            <SummaryRow label="Subtotal" value={subtotal} />
            <Field label="Tax">
              <Input
                className="h-9 text-sm"
                type="number"
                value={form.tax}
                onChange={e => setForm(f => ({ ...f, tax: Number(e.target.value) }))}
              />
            </Field>
            <Field label="Discount">
              <Input
                className="h-9 text-sm"
                type="number"
                value={form.discount}
                onChange={e => setForm(f => ({ ...f, discount: Number(e.target.value) }))}
              />
            </Field>
            <SummaryRow label="Total" value={total} strong />
          </div>
        </div>

        {/* Attachment */}
        <div className="px-4 pb-4">
          <Field label="Attachments">
            <Input
              type="file"
              onChange={e => setForm(f => ({ ...f, attach: e.target.files?.[0] || null }))}
            />
          </Field>
        </div>

        {/* Footer */}
        <div className="flex gap-2 border-t p-4">
          <Button variant="outline" onClick={() => router.back()}>Cancel</Button>
          <SubmitButton onClick={save} disabled={!canSave}>Save Bill</SubmitButton>
        </div>
      </div>
    </div>
  )
}

function Field({ label, className, children }: React.PropsWithChildren<{ label: string; className?: string }>) {
  return (
    <div className={className}>
      <label className="mb-1 block text-sm font-semibold">{label}</label>
      {children}
    </div>
  )
}

function SummaryRow({ label, value, strong }: { label: string; value: number; strong?: boolean }) {
  return (
    <div className="flex items-center justify-between rounded-md border bg-muted/30 px-3 py-2">
      <span className="text-sm">{label}:</span>
      <span className={`tabular-nums ${strong ? "font-semibold" : ""}`}>{value.toFixed(2)}</span>
    </div>
  )
}
